/**
 * Copyright (c) 2016-2018 TypeFox and others.
 * 
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License v. 2.0 which is available at
 * http://www.eclipse.org/legal/epl-2.0,
 * or the Eclipse Distribution License v. 1.0 which is available at
 * http://www.eclipse.org/org/documents/edl-v10.php.
 * 
 * SPDX-License-Identifier: EPL-2.0 OR BSD-3-Clause
 */
package org.eclipse.lsp4j;

import org.eclipse.lsp4j.UnchangedDocumentDiagnosticReport;
import org.eclipse.lsp4j.jsonrpc.validation.NonNull;
import org.eclipse.lsp4j.util.Preconditions;
import org.eclipse.xtext.xbase.lib.Pure;
import org.eclipse.xtext.xbase.lib.util.ToStringBuilder;

/**
 * An unchanged document diagnostic report for a workspace diagnostic result.
 * <p>
 * Since 3.17.0
 */
@SuppressWarnings("all")
public class WorkspaceUnchangedDocumentDiagnosticReport extends UnchangedDocumentDiagnosticReport {
  /**
   * The URI for which diagnostic information is reported.
   */
  @NonNull
  private String uri;
  
  /**
   * The version number for which the diagnostics are reported.
   * If the document is not marked as open `null` can be provided.
   */
  private Integer version;
  
  public WorkspaceUnchangedDocumentDiagnosticReport() {
  }
  
  public WorkspaceUnchangedDocumentDiagnosticReport(@NonNull final String resultId, @NonNull final String uri, final Integer version) {
    super(resultId);
    this.uri = Preconditions.<String>checkNotNull(uri, "uri");
    this.version = version;
  }
  
  /**
   * The URI for which diagnostic information is reported.
   */
  @Pure
  @NonNull
  public String getUri() {
    return this.uri;
  }
  
  /**
   * The URI for which diagnostic information is reported.
   */
  public void setUri(@NonNull final String uri) {
    this.uri = Preconditions.checkNotNull(uri, "uri");
  }
  
  /**
   * The version number for which the diagnostics are reported.
   * If the document is not marked as open `null` can be provided.
   */
  @Pure
  public Integer getVersion() {
    return this.version;
  }
  
  /**
   * The version number for which the diagnostics are reported.
   * If the document is not marked as open `null` can be provided.
   */
  public void setVersion(final Integer version) {
    this.version = version;
  }
  
  @Override
  @Pure
  public String toString() {
    ToStringBuilder b = new ToStringBuilder(this);
    b.add("uri", this.uri);
    b.add("version", this.version);
    b.add("kind", getKind());
    b.add("resultId", getResultId());
    return b.toString();
  }
  
  @Override
  @Pure
  public boolean equals(final Object obj) {
    if (this == obj)
      return true;
    if (obj == null)
      return false;
    if (getClass() != obj.getClass())
      return false;
    if (!super.equals(obj))
      return false;
    WorkspaceUnchangedDocumentDiagnosticReport other = (WorkspaceUnchangedDocumentDiagnosticReport) obj;
    if (this.uri == null) {
      if (other.uri != null)
        return false;
    } else if (!this.uri.equals(other.uri))
      return false;
    if (this.version == null) {
      if (other.version != null)
        return false;
    } else if (!this.version.equals(other.version))
      return false;
    return true;
  }
  
  @Override
  @Pure
  public int hashCode() {
    final int prime = 31;
    int result = super.hashCode();
    result = prime * result + ((this.uri== null) ? 0 : this.uri.hashCode());
    return prime * result + ((this.version== null) ? 0 : this.version.hashCode());
  }
}
