# (c) Copyright 2009-2021. CodeWeavers, Inc.
from __future__ import division

from gi.repository import Gtk

import cxproduct
import cxobservable

import cxaiemedia

import cxguitools
import distversion

from cxutils import _

# These are the observable events
DIALOG_CLOSED = 'dialog_closed'


#
# Dialog instance
#

class PreferenceDialogController(cxobservable.Object):

    observable_events = frozenset((DIALOG_CLOSED,))

    def __init__(self, transient_for):
        cxobservable.Object.__init__(self)

        #-----------------------------------------------------------------------
        #  Setup the GUI
        #-----------------------------------------------------------------------
        self.xml = Gtk.Builder()
        self.xml.set_translation_domain("crossover")
        self.xml.add_from_file(cxguitools.get_ui_path("cxprefs"))
        # Set runtime labels
        self.xml.get_object("applyCxfixesCheckbox").set_label(_("Install recommended %(platform)s packages") % {'platform': distversion.PLATFORM})

        # Set initial values
        askForRatings = True
        autoUpdate = True
        checkForUpdates = True
        showUntested = True
        applyCxfixes = True

        global_config = cxproduct.get_config()
        if global_config['CrossOver'].get('AskForRatings', '1') == '0':
            askForRatings = False

        if global_config['CrossOver'].get('CheckForUpdates', '1') == '0':
            checkForUpdates = False

        if global_config['OfficeSetup'].get('AutoUpdate', '1') == '0':
            autoUpdate = False

        if global_config['OfficeSetup'].get('ShowUntestedApps', '1') != '1':
            showUntested = False

        if global_config['OfficeSetup'].get('ApplyCxfixes', '1') != '1':
            applyCxfixes = False

        self.xml.get_object('ratingCheckbox').set_active(not askForRatings)
        self.xml.get_object('checkForUpdatesCheckbox').set_active(checkForUpdates)
        self.xml.get_object('downloadCheckbox').set_active(autoUpdate)
        self.xml.get_object('showUntestedCheckbox').set_active(showUntested)
        self.xml.get_object('applyCxfixesCheckbox').set_active(applyCxfixes)

        # Connect the signals after setting the default values so they don't get
        # set in the config and can be changed later.
        self.xml.connect_signals(self)

        self.update_cache_label()

        dialog = self.xml.get_object("PrefsDialog")
        dialog.props.transient_for = transient_for
        dialog.show()

    def update_cache_label(self):
        size = cxaiemedia.size_cached_installers()

        self.xml.get_object('ClearCacheButton').set_sensitive(size != 0)

        if size == 0:
            message = _("There are currently no downloaded installers saved locally.")
        elif size < 1024 * 1024:
            message = _("Downloaded installers are currently using %0.1f KB of disk space.") % (size / 1024)
        elif size < 1024 * 1024 * 1024:
            message = _("Downloaded installers are currently using %0.1f MB of disk space.") % (size / (1024 * 1024))
        else:
            message = _("Downloaded installers are currently using %0.1f GB of disk space.") % (size / (1024 * 1024 * 1024))

        self.xml.get_object('InstallerCacheLabel').set_text(message)

    def on_ClearCacheButton_clicked(self, _widget):
        cxaiemedia.delete_cached_installers()

        self.update_cache_label()

    def downloadCheckbox_toggled_cb(self, caller):
        # pylint: disable=R0201
        if caller.get_active():
            cxproduct.save_setting('OfficeSetup', 'AutoUpdate', '1')
        else:
            cxproduct.save_setting('OfficeSetup', 'AutoUpdate', '0')

    def ratingCheckbox_toggled_cb(self, caller):
        # pylint: disable=R0201
        if caller.get_active():
            cxproduct.save_setting('CrossOver', 'AskForRatings', '0')
        else:
            cxproduct.save_setting('CrossOver', 'AskForRatings', '1')

    def checkForUpdatesCheckbox_toggled_cb(self, caller):
        # pylint: disable=R0201
        if caller.get_active():
            cxproduct.save_setting('CrossOver', 'CheckForUpdates', '1')
        else:
            cxproduct.save_setting('CrossOver', 'CheckForUpdates', '0')

    def untestedCheckbox_toggled_cb(self, caller):
        # pylint: disable=R0201
        if caller.get_active():
            cxproduct.save_setting('OfficeSetup', 'ShowUntestedApps', '1')
        else:
            cxproduct.save_setting('OfficeSetup', 'ShowUntestedApps', '0')

    def on_applyCxfixesCheckbox_toggled(self, caller):
        # pylint: disable=R0201
        if caller.get_active():
            cxproduct.save_setting('OfficeSetup', 'ApplyCxfixes', '1')
        else:
            cxproduct.save_setting('OfficeSetup', 'ApplyCxfixes', '0')

    def quit_requested(self, *_args):
        self.xml.get_object("PrefsDialog").destroy()
        self.emit_event(DIALOG_CLOSED)
        cxguitools.toplevel_quit()

    def present(self, transient_for):
        dialog = self.xml.get_object("PrefsDialog")
        dialog.props.transient_for = transient_for
        dialog.present()


#
# Global dialog management
#

DIALOG = None

def _dialog_closed(_dialog, _event, _data):
    # pylint: disable=W0603
    global DIALOG
    DIALOG = None

def open_or_show(transient_for):
    # pylint: disable=W0603
    global DIALOG
    if DIALOG is None:
        DIALOG = PreferenceDialogController(transient_for)
        DIALOG.add_observer(DIALOG_CLOSED, _dialog_closed)
    else:
        DIALOG.present(transient_for)
    return DIALOG
